<?php
// daily_work_copy.php - Generate WhatsApp text format
header('Content-Type: application/json');

ini_set('display_errors', 1);
error_reporting(E_ALL);

include 'config.php';
include 'session.php';

$daily_work_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$daily_work_id) {
    echo json_encode(['success' => false, 'message' => 'ID daily work tidak valid']);
    exit();
}

try {
    // Get daily work data
    $sql_work = "
        SELECT 
            dw.*,
            COUNT(DISTINCT dwp.id) as total_plans,
            COUNT(DISTINCT dwr.id) as total_results,
            COUNT(DISTINCT CASE WHEN dwr.status = 'completed' THEN dwr.id END) as completed_results
        FROM daily_work dw
        LEFT JOIN daily_work_plans dwp ON dw.id = dwp.daily_work_id
        LEFT JOIN daily_work_results dwr ON dw.id = dwr.daily_work_id
        WHERE dw.id = ?
        GROUP BY dw.id
    ";

    $stmt_work = mysqli_prepare($conn, $sql_work);
    mysqli_stmt_bind_param($stmt_work, "i", $daily_work_id);
    mysqli_stmt_execute($stmt_work);
    $result_work = mysqli_stmt_get_result($stmt_work);
    $daily_work = mysqli_fetch_assoc($result_work);

    if (!$daily_work) {
        echo json_encode(['success' => false, 'message' => 'Daily work tidak ditemukan']);
        exit();
    }

    // Get plans
    $sql_plans = "SELECT * FROM daily_work_plans WHERE daily_work_id = ? ORDER BY urutan ASC, id ASC";
    $stmt_plans = mysqli_prepare($conn, $sql_plans);
    mysqli_stmt_bind_param($stmt_plans, "i", $daily_work_id);
    mysqli_stmt_execute($stmt_plans);
    $result_plans = mysqli_stmt_get_result($stmt_plans);
    $plans = [];
    while ($row = mysqli_fetch_assoc($result_plans)) {
        $plans[] = $row;
    }

    // Get results
    $sql_results = "SELECT * FROM daily_work_results WHERE daily_work_id = ? ORDER BY urutan ASC, id ASC";
    $stmt_results = mysqli_prepare($conn, $sql_results);
    mysqli_stmt_bind_param($stmt_results, "i", $daily_work_id);
    mysqli_stmt_execute($stmt_results);
    $result_results = mysqli_stmt_get_result($stmt_results);
    $results = [];
    while ($row = mysqli_fetch_assoc($result_results)) {
        $results[] = $row;
    }

    // Indonesian day names
    $day_names = [
        'Sunday' => 'Minggu', 'Monday' => 'Senin', 'Tuesday' => 'Selasa', 'Wednesday' => 'Rabu',
        'Thursday' => 'Kamis', 'Friday' => 'Jumat', 'Saturday' => 'Sabtu'
    ];
    $day_name = $day_names[date('l', strtotime($daily_work['tanggal']))];

    // Calculate work duration
    $work_duration = '';
    if ($daily_work['jam_masuk'] && $daily_work['jam_pulang']) {
        $start = new DateTime($daily_work['jam_masuk']);
        $end = new DateTime($daily_work['jam_pulang']);
        $diff = $start->diff($end);
        $work_duration = $diff->h . ' jam ' . $diff->i . ' menit';
    }

    // Status labels
    $status_labels = [
        'completed' => '✅ Selesai',
        'in_progress' => '🔄 Dalam Proses',
        'pending' => '⏳ Pending',
        'cancelled' => '❌ Dibatalkan'
    ];

    // Generate WhatsApp text
    $wa_text = "*📋 DAILY WORK REPORT*\n";
    $wa_text .= "*CV. Sumber Berkat Anak*\n\n";
    
    // Basic info
    $wa_text .= "📅 *Tanggal:* " . $day_name . ", " . date('d F Y', strtotime($daily_work['tanggal'])) . "\n";
    $wa_text .= "👤 *Nama:* " . $daily_work['created_by'] . "\n";
    
    if ($daily_work['jam_masuk'] || $daily_work['jam_pulang']) {
        $wa_text .= "🕐 *Jam Kerja:* ";
        if ($daily_work['jam_masuk']) {
            $wa_text .= date('H:i', strtotime($daily_work['jam_masuk']));
        }
        if ($daily_work['jam_pulang']) {
            $wa_text .= " - " . date('H:i', strtotime($daily_work['jam_pulang']));
        }
        if ($work_duration) {
            $wa_text .= " (" . $work_duration . ")";
        }
        $wa_text .= "\n";
    }
    
    $wa_text .= "\n";
    
    // Summary
    $completion_rate = $daily_work['total_plans'] > 0 ? 
        round(($daily_work['completed_results'] / $daily_work['total_plans']) * 100) : 0;
    
    $wa_text .= "📊 *Summary:*\n";
    $wa_text .= "• Plan: " . $daily_work['total_plans'] . " items\n";
    $wa_text .= "• Hasil: " . $daily_work['total_results'] . " items\n";
    $wa_text .= "• Progress: " . $completion_rate . "%\n\n";
    
    // Plans & Results
    if (!empty($plans)) {
        $wa_text .= "📝 *PLAN & HASIL PEKERJAAN:*\n\n";
        
        foreach ($plans as $index => $plan) {
            $wa_text .= ($index + 1) . ". *" . $plan['subject'] . "*\n";
            
            if ($plan['detail']) {
                $wa_text .= "   📋 " . $plan['detail'] . "\n";
            }
            
            // Find related result
            $plan_result = null;
            foreach ($results as $result) {
                if ($result['plan_id'] == $plan['id']) {
                    $plan_result = $result;
                    break;
                }
            }
            
            if ($plan_result) {
                $status_text = $status_labels[$plan_result['status']] ?? $plan_result['status'];
                $wa_text .= "   " . $status_text . "\n";
                if ($plan_result['detail']) {
                    $wa_text .= "   💬 " . $plan_result['detail'] . "\n";
                }
            } else {
                $wa_text .= "   ⏳ Belum dikerjakan\n";
            }
            
            $wa_text .= "\n";
        }
    }
    
    // Additional results
    $additional_results = array_filter($results, function($result) {
        return $result['plan_id'] === null || $result['plan_id'] == '';
    });
    
    if (!empty($additional_results)) {
        $wa_text .= "➕ *HASIL TAMBAHAN:*\n\n";
        
        $additional_index = 1;
        foreach ($additional_results as $result) {
            $status_text = $status_labels[$result['status']] ?? $result['status'];
            $wa_text .= $additional_index . ". *" . $result['subject'] . "* " . $status_text . "\n";
            
            if ($result['detail']) {
                $wa_text .= "   💬 " . $result['detail'] . "\n";
            }
            
            $wa_text .= "\n";
            $additional_index++;
        }
    }
    
    // Footer
    $wa_text .= "---\n";
    $wa_text .= "🏢 CV. Sumber Berkat Anak\n";
    $wa_text .= "📱 Generated: " . date('d/m/Y H:i') . " WIB";

    echo json_encode([
        'success' => true, 
        'whatsapp_text' => $wa_text,
        'summary' => [
            'total_plans' => $daily_work['total_plans'],
            'total_results' => $daily_work['total_results'],
            'completion_rate' => $completion_rate
        ]
    ]);

} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>